<?php

use Illuminate\Support\Facades\Facade;
use function Safe\date;
use function Safe\scandir;

/**
 * Given a .env config constant, retrieve the env value and remove any trailing /.
 *
 * @param string      $cst     constant to fetch
 * @param string|null $default default value if does not exists
 *
 * @return string trimmed result
 */
if (!function_exists('renv')) {
	function renv(string $cst, ?string $default = null): string
	{
		return rtrim((string) (env($cst, $default) ?? ''), '/');
	}
}

return [
	/*
	|--------------------------------------------------------------------------
	| Application Name
	|--------------------------------------------------------------------------
	|
	| This value is the name of your application. This value is used when the
	| framework needs to place the application's name in a notification or
	| any other location as required by the application or its packages.
	|
	*/

	'name' => env('APP_NAME', 'Lychee'),

	/*
	|--------------------------------------------------------------------------
	| Application Environment
	|--------------------------------------------------------------------------
	|
	| This value determines the "environment" your application is currently
	| running in. This may determine how you prefer to configure various
	| services your application utilizes. Set this in your ".env" file.
	|
	*/

	'env' => env('APP_ENV', 'production'),

	/*
	|--------------------------------------------------------------------------
	| Application Debug Mode
	|--------------------------------------------------------------------------
	|
	| When your application is in debug mode, detailed error messages with
	| stack traces will be shown on every error that occurs within your
	| application. If disabled, a simple generic error page is shown.
	|
	*/

	'debug' => (bool) env('APP_DEBUG', false),

	/*
	|--------------------------------------------------------------------------
	| Application URL
	|--------------------------------------------------------------------------
	|
	| This URL is used by the console to properly generate URLs when using
	| the Artisan command line tool. You should set this to the root of
	| your application so that it is used when running Artisan tasks.
	|
	| url : the base url of your Lychee install up to the tld (end '/' will be trimmed)
	| dir_url : the path of your Lychee install from the tld (will be prefixed by '/' and end '/' will be trimmed)
	|
	| asset_url : should be left to default (null).
	*/

	'url' => renv('APP_URL', 'http://localhost'),

	'dir_url' => env('APP_DIR', '') === '' ? '' : ('/' . trim((string) (env('APP_DIR') ?? ''), '/')),

	'asset_url' => null,

	/*
	|--------------------------------------------------------------------------
	| Application Timezone
	|--------------------------------------------------------------------------
	|
	| Here you may specify the default timezone for your application, which
	| will be used by the PHP date and date-time functions. We have gone
	| ahead and set this to a sensible default for you out of the box.
	|
	*/

	'timezone' => env('TIMEZONE', date('e')),

	/*
	|--------------------------------------------------------------------------
	| Application Locale Configuration
	|--------------------------------------------------------------------------
	|
	| The application locale determines the default locale that will be used
	| by the translation service provider. You are free to set this value
	| to any of the locales which will be supported by the application.
	|
	*/

	'locale' => 'en',

	/*
	|--------------------------------------------------------------------------
	| Application Fallback Locale
	|--------------------------------------------------------------------------
	|
	| The fallback locale determines the locale to use when the current one
	| is not available. You may change the value to correspond to any of
	| the language folders that are provided through your application.
	|
	*/

	'fallback_locale' => 'en',

	/*
	|--------------------------------------------------------------------------
	| Application Avilable Locale
	|--------------------------------------------------------------------------
	|
	| List of locale supported by Lychee.
	| ['cz', 'de', 'el', 'en', 'es', 'fr', 'it', 'nl', 'no', 'pl', 'pt', 'ru', 'sk', 'sv', 'vi', 'zh_CN', 'zh_TW']
	*/

	'supported_locale' => array_diff(scandir(base_path('lang')), ['..', '.']),

	/*
	|--------------------------------------------------------------------------
	| Faker Locale
	|--------------------------------------------------------------------------
	|
	| This locale will be used by the Faker PHP library when generating fake
	| data for your database seeds. For example, this will be used to get
	| localized telephone numbers, street address information and more.
	|
	*/

	'faker_locale' => 'en_US',

	/*
	|--------------------------------------------------------------------------
	| Skip diagnostics checks
	|--------------------------------------------------------------------------
	|
	| Allows to define class names of diagnostics checks that will be skipped.
	|
	*/
	'skip_diagnostics_checks' => explode(',', (string) env('SKIP_DIAGNOSTICS_CHECKS', '')),

	/*
	|--------------------------------------------------------------------------
	| Encryption Key
	|--------------------------------------------------------------------------
	|
	| This key is used by the Illuminate encrypter service and should be set
	| to a random, 32 character string, otherwise these encrypted strings
	| will not be safe. Please do this before deploying an application!
	|
	*/

	'key' => env('APP_KEY'),

	'cipher' => env('APP_CIPHER', 'AES-256-CBC'),

	/*
	|--------------------------------------------------------------------------
	| Maintenance Mode Driver
	|--------------------------------------------------------------------------
	|
	| These configuration options determine the driver used to determine and
	| manage Laravel's "maintenance mode" status. The "cache" driver will
	| allow maintenance mode to be controlled across multiple machines.
	|
	| Supported drivers: "file", "cache"
	|
	*/

	'maintenance' => [
		'driver' => 'file',
		// 'store'  => 'redis',
	],

	/*
	|--------------------------------------------------------------------------
	| Autoloaded Service Providers
	|--------------------------------------------------------------------------
	|
	| The service providers listed here will be automatically loaded on the
	| request to your application. Feel free to add your own services to
	| this array to grant expanded functionality to your applications.
	|
	*/

	'providers' => [
		/*
		 * Laravel Framework Service Providers...
		 */
		Illuminate\Auth\AuthServiceProvider::class,
		Illuminate\Broadcasting\BroadcastServiceProvider::class,
		Illuminate\Bus\BusServiceProvider::class,
		Illuminate\Cache\CacheServiceProvider::class,
		Illuminate\Foundation\Providers\ConsoleSupportServiceProvider::class,
		Illuminate\Cookie\CookieServiceProvider::class,
		Illuminate\Database\DatabaseServiceProvider::class,
		Illuminate\Encryption\EncryptionServiceProvider::class,
		Illuminate\Filesystem\FilesystemServiceProvider::class,
		Illuminate\Foundation\Providers\FoundationServiceProvider::class,
		Illuminate\Hashing\HashServiceProvider::class,
		Illuminate\Mail\MailServiceProvider::class,
		Illuminate\Notifications\NotificationServiceProvider::class,
		Illuminate\Pagination\PaginationServiceProvider::class,
		Illuminate\Pipeline\PipelineServiceProvider::class,
		Illuminate\Queue\QueueServiceProvider::class,
		Illuminate\Redis\RedisServiceProvider::class,
		Illuminate\Session\SessionServiceProvider::class,
		Illuminate\Translation\TranslationServiceProvider::class,
		Illuminate\Validation\ValidationServiceProvider::class,
		Illuminate\View\ViewServiceProvider::class,

		/*
		 * Package Service Providers...
		 */

		\SocialiteProviders\Manager\ServiceProvider::class,
		// Barryvdh\Debugbar\ServiceProvider::class,
		Mavinoo\Batch\BatchServiceProvider::class,

		/*
		 * Application Service Providers...
		 */
		App\Providers\AppServiceProvider::class,
		App\Providers\AuthServiceProvider::class,
		App\Providers\EventServiceProvider::class,
		App\Providers\RouteServiceProvider::class,
		LycheeVerify\VerifyServiceProvider::class,
	],

	/*
	|--------------------------------------------------------------------------
	| Class Aliases
	|--------------------------------------------------------------------------
	|
	| This array of class aliases will be registered when this application
	| is started. However, feel free to register as many as you wish as
	| the aliases are "lazy" loaded so they don't hinder performance.
	|
	*/

	'aliases' => Facade::defaultAliases()->merge([
		'DebugBar' => Barryvdh\Debugbar\Facades\Debugbar::class,
		'Helpers' => App\Facades\Helpers::class,
		'Features' => App\Assets\Features::class,
		// Aliases for easier access in the blade templates
		'Configs' => App\Models\Configs::class,
		'AlbumPolicy' => App\Policies\AlbumPolicy::class,
		'PhotoPolicy' => App\Policies\PhotoPolicy::class,
		'SettingsPolicy' => App\Policies\SettingsPolicy::class,
		'UserPolicy' => App\Policies\UserPolicy::class,
		'User' => App\Models\User::class,
		'SizeVariantType' => App\Enum\SizeVariantType::class,
		'FileStatus' => App\Enum\FileStatus::class,
		'PhotoLayoutType' => \App\Enum\PhotoLayoutType::class,
	])->toArray(),
];
