<?php

namespace Laragear\WebAuthn\Migrations;

use Illuminate\Database\Schema\Blueprint;
use Laragear\MetaModel\CustomizableMigration;
use Laragear\WebAuthn\Enums\Formats;

class WebAuthnAuthenticationMigration extends CustomizableMigration
{
    /**
     * @inheritDoc
     */
    public function create(Blueprint $table): void
    {
        // Here we set the PublicKeyCredential ID generated by the authenticator as string.
        // This way it's easier and faster for the database to find the right credential
        // on the Assertion procedure as the device returns which credential it used.
        $table->string('id', 510)->primary();

        $this->createMorph($table, 'authenticatable', 'webauthn_user_index');

        // When requesting to create a credential, the app will set a "user handle" to be
        // a UUID to anonymize the user personal information. If a second credential is
        // created, the first UUID is copied to the new one, keeping the association.
        $table->uuid('user_id');

        // The app may allow the user to name or rename a credential to a friendly name,
        // like "John's iPhone" or "Office Computer". This column is nullable, so it's
        // up to the app to use an alias. Otherwise, the app can use custom columns.
        $table->string('alias')->nullable();

        // Allows to detect cloned credentials when these do not share the same counter.
        $table->unsignedBigInteger('counter')->nullable();
        // Who created the credential? Should be the same reported by the Authenticator.
        $table->string('rp_id');
        // Where the credential was created? Should be the same reported by the Authenticator.
        $table->string('origin');
        // The available "ways to transmit" the public key between the browser and Authenticator.
        // It may be generated by the authenticator when it creates it, that's why is nullable.
        // On assertion, this will allow the authenticator where to look for the private key.
        $table->json('transports')->nullable();
        // The "type" or "properties" of the authenticator. Sometimes these are zeroes or null.
        $table->uuid('aaguid')->nullable(); // GUID are essentially UUID

        // This is the public key the server will use to verify the challenges are corrected.
        $table->text('public_key');
        // The attestation of the public key.
        $table->string('attestation_format')->default(Formats::NONE->value);
        // This would hold the certificate chain for other different attestation formats.
        $table->json('certificates')->nullable();

        // Run the additional migration instructions...
        $this->addColumns($table);

        // A way to disable the credential without deleting it.
        $table->timestamp('disabled_at')->nullable();
        $table->timestamps();
    }
}
